/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#ifndef SIMULATE_H_
#define SIMULATE_H_

#ifdef MSDOS
 #include "pstdint.h"
#else
 #include <stdint.h>
#endif
#include <stdbool.h>
#include "game.h"

/* Utility functions for simulating "what-if" game scenarios. These build upon
 * the primitive game_copy() provided by the game. */

typedef struct simulator_s simulator;

/* Initialise a simulator (pedantically, a simulated state). It will take a
 * copy of the game. May return false upon memory exhaustion. */
bool sim_init(simulator* sim, const fcgame* game);

/* Copy a simulated state. This also involves copying the game. May return
 * false upon memory exhaustion. Deepen provides this combined with move. */
bool sim_copy(const simulator* src, simulator* dst);

/* Destroy a simulator. You must do this to ensure that memory used within the
 * simulator is freed. */
void sim_destroy(simulator* sim);

/* Access the game being simulated so that you can sniff its state. Do NOT
 * attempt to run or otherwise modify the game, else bad things may happen.
 * Will return null if the game has been won by this stage. */
const fcgame* sim_game(simulator* sim);

/* Returns true if the simulated game has been won. */
bool sim_won(const simulator* sim);

/* Return the current player number of the simulated game (who shall move).
 * If the game has been won, this is the winner. */
uint8_t sim_player(const simulator* sim);

/* Simulate the current player making a given move in-place.
 * Returns false if move is invalid. */
bool sim_move(simulator* sim, uint8_t x, uint8_t y);

/* Simulate the current player making a given move. Initialises 'next' with the
 * result. Do not provide the same simulator as both sim and the next target,
 * and remember to destroy both before discarding or overwriting them.
 * Returns false, and leaves next undefined, if move is invalid or memory is
 * exhausted. If you need to tell the difference, you'll have to use game_valid
 * on this move beforehand (via sim_game). */
bool sim_deepen(const simulator* sim, simulator* next, uint8_t x, uint8_t y);

/******************************************************************************/
/* The simulator structure is opaque. However, to allow you to stack-allocate
 * them, the structure is here, so that the compiler knows how big it is. */
struct simulator_s {
	fcgame* game;
	bool place; /* Perform a placement. */
	uint8_t p_x, p_y; /* Place at these co-ordinates. */
	bool invalid; /* Placement was invalid */
	bool won; /* Game has been won. */
	uint8_t wonby; /* ID of the player which won. */
	fccallbacks callbacks;
};
/* Resist the temptation to manipulate this structure yourself. */
/******************************************************************************/

#endif

