/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#include <stdlib.h>
#include "simulate.h"

static void sim_cb_place(void* data, uint8_t player, uint8_t* x, uint8_t* y) {
	simulator* sim = (simulator*) data;
	if(sim->place) {
		*x = sim->p_x;
		*y = sim->p_y;
		sim->invalid = false;
		sim->place = false; /* Break out at start of next turn */
	} else { game_break(sim->game); } /* Clean place to stop simulation */
}

static void sim_cb_placed(void* data, uint8_t player, uint8_t x, uint8_t y) {}

static void sim_cb_invalid(void* data, uint8_t player, uint8_t x, uint8_t y) {
	simulator* sim = (simulator*) data;
	sim->invalid = true;
	game_break(sim->game); /* Break here; turn has NOT advanced */
}

static void sim_cb_full(void* data, uint8_t player, uint8_t x, uint8_t y) {}
static void sim_cb_explode(void* data, const fcexplosion* explosion) {}
static void sim_cb_eliminated(void* data, uint8_t player) {}

static void sim_cb_wins(void* data, uint8_t player) {
	simulator* sim = (simulator*) data;
	sim->won = true;
	sim->wonby = player;
	/* Game will now auto-abort when this callback returns */
}

bool sim_init(simulator* sim, const fcgame* game) {
	sim->callbacks.data = sim;
	sim->callbacks.place      = sim_cb_place;
	sim->callbacks.placed     = sim_cb_placed;
	sim->callbacks.invalid    = sim_cb_invalid;
	sim->callbacks.full       = sim_cb_full; 
	sim->callbacks.explode    = sim_cb_explode;
	sim->callbacks.eliminated = sim_cb_eliminated;
	sim->callbacks.wins       = sim_cb_wins;
	if((sim->game = malloc(sizeof(fcgame)))) {
		if(game_copy(game, sim->game, &sim->callbacks)) {
			sim->place = false;
			sim->p_x = sim->p_y = 255;
			sim->invalid = false;
			sim->wonby = 255;
			/* Initialisation which actually matters */
			sim->won = false;
			return true;
		}
		free(sim->game);
	}
	return false;
}

bool sim_copy(const simulator* src, simulator* dst) {
	if(src->won) {
		dst->won = true;
		dst->wonby = src->wonby;
		dst->game = 0;
		return true; /* No allocation required */
	} else {
		return sim_init(dst, src->game);
	}
}

void sim_destroy(simulator* sim) {
	if(!sim->won) { /* Won games deallocate themselves. */
		game_abort(sim->game);
		game_play(sim->game);
		/* Will break out and dealloc immediately. */
	}
	if(sim->game) { /* The game alloc'd the grid; we alloc'd the game. */
		free(sim->game);
	}
}

const fcgame* sim_game(simulator* sim) {
	return sim->game;
}

bool sim_won(const simulator* sim) {
	return sim->won;
}

uint8_t sim_player(const simulator* sim) {
	if(sim->won) { return sim->wonby; }
	return game_curplayer(sim->game);
}

bool sim_move(simulator* sim, uint8_t x, uint8_t y) {
	/* First things first, no moves are valid for already-won games. */
	if(sim->won) { return false; }
	/* Set up our state so that the callbacks know to make a move. */
	sim->place = true;
	sim->p_x = x;
	sim->p_y = y;
	/* Run the game. Will run for zero or one turns before break or win. */
	game_play(sim->game);
	/* Return move validity. If valid, one turn occured; else, zero. */
	return !sim->invalid;
}

bool sim_deepen(const simulator* sim, simulator* next, uint8_t x, uint8_t y) {
	if(sim_copy(sim, next)) {
		/* Move on the copy; do not reach OOM code */
		if(sim_move(next, x, y)) {
			return true;
		} else {
			sim_destroy(next);
			return false;
		}
	}
	/* Out of memory */
	return false;
}

