/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef MSDOS
 #include "doshacks.h"
#endif
#if defined(MSDOS) || defined(WIN32)
 #include <curses.h>
inline static char* getlogin(void) {
	return "Player";
}
#else
 #include <ncurses.h>
#endif
#include "game.h"
#include "ai.h"
#include "ncursesui.h"

static void do_gpl_waffle(void);

/** Configuring the game ******************************************************/

static bool playing_again = false;

/* Asks question; writes to answer (up to answerlen bytes, always terminated) */
static void ask(const char* question, const char* defvalue, char* answer, int answerlen) {
	printw("%s [%s]? ", question, defvalue);
	refresh();
	getnstr(answer, answerlen);
	if(answer[0] == '\0') {
		strncpy(answer, defvalue, answerlen);
	}
	answer[answerlen-1] = '\0';
}

/* Converts string to number; returns true if format is good and in range. */
static bool parsenum(const char* string, long* number, long low, long high) {
	char* end;
	*number = strtol(string, &end, 10);
	if(*string == '\0' || *end != '\0' ||
	     *number < low || *number > high) {
		return false;
	}
	return true;
}
/* Wrapper for dealing with uint8_ts (which is actually all we ever do) */
static bool parsenum_u8(const char* string, uint8_t* number,
	uint8_t low, uint8_t high) {

	long largenum;
	if(parsenum(string, &largenum, low, high))
		{ *number = largenum; return true; }
	return false;
}

/* Configure an AI player for the given player number, and add to the setup.
 * If multiple AI implementations exist in the future, this could include
 * prompting to ask which one to use, or configuration such as search depth. */
static void make_ai(setupinfo* gamesetup, uint8_t player) {
	strcpy(gamesetup->player_name[player], "Computer");
	gamesetup->player_ai[player] = true;
}

bool setup_game(setupinfo* gamesetup) {
	char prompt[32];
	char defvalue[16];
	char answers[16];
	uint8_t players;
	uint8_t minheight;

	clear();
	scrollok(stdscr, TRUE);
	do_gpl_waffle();
	echo();

	if(playing_again) {
		ask("Play again", "n", answers, 2);
		if(answers[0] == 'n' || answers[0] == 'N')
			{ return false; }
	}

	do {
		printw(
		"There are six player colours available.\n"
		"For each player, give a name; or 'C' for a computer; or a space for nobody.\n"
		"\n");

		/* Avoid trampling gamesetup->players until been through once;
		 * this allows defaulting to previous after the first run. */
		players = 0;
		for(int pslot = 0; pslot < 6; pslot++) {
			int colour = COLOR_PAIR(COLPLAY(pslot));
			snprintf(prompt, 32, "Colour %d", pslot + 1);
			if(playing_again) {
				strcpy(defvalue, " ");
				for(int p = 0; p < gamesetup->players; p++) {
					/* Already bounds-checked these */
					if(gamesetup->player_color[p] == pslot)
						{ strcpy(defvalue,
							gamesetup->player_ai[p]
							? "C" :
							gamesetup->
							player_name[p]); }
				}
			} else { switch(pslot) {
				case 0:
					strncpy(defvalue, getlogin(), 15);
					if(defvalue[0] == '\0')
						{ strcpy(defvalue, "You"); }
					defvalue[15] = '\0';
					break;
				case 2: strcpy(defvalue, "C"); break;
				default: strcpy(defvalue, " ");
			}}
			attron( colour);
			ask(prompt, defvalue, answers, 16);
			attroff(colour);
			if(answers[0] != ' ' || answers[1] != '\0') {
				gamesetup->player_color[players] = pslot;
				if(answers[0] == 'C' && answers[1] == '\0') {
					make_ai(gamesetup, players);
				} else {
				/* Bounds of string checked already in ask() */
					strcpy(gamesetup->player_name[players],
						answers);
					gamesetup->player_ai[players] = false;
				}
				players++;
			}
		}
		/* Even if insufficient players, we have now trampled the
		 * gamesetup structure, and MUST keep it consistent! */
		gamesetup->players = players;

		if(gamesetup->players < 2) {
			printw("\n  FreeChain is a game for at least two players.\n\n");
			refresh();
		}
	} while(gamesetup->players < 2);

	do {	snprintf(defvalue, 64, "%d",
			playing_again ? gamesetup->width :
			                DEFAULT_GRID_WIDTH);
		ask("Grid width  (1 to 255)", defvalue, answers, 4);
	} while(!parsenum_u8(answers, &gamesetup->width,  1, 255));

	/* Grid size must at least cover all players.
	 * W x H >= P --> H >= P / W, ceiling */
	minheight = gamesetup->players / gamesetup->width;
	if         (gamesetup->players % gamesetup->width) { minheight++; }

	snprintf(prompt, 32, "Grid height (%d to 255)", (int) minheight);
	do {	snprintf(defvalue, 64, "%d",
			playing_again ? gamesetup->height :
			                DEFAULT_GRID_HEIGHT);
		ask(prompt, defvalue, answers, 4);
	} while(!parsenum_u8(answers, &gamesetup->height, minheight, 255));

	noecho();
	scrollok(stdscr, FALSE);
	clear();
	refresh();
	playing_again = true;
	return true;
}

/* Ideally, want something like this: */
/* +-- FreeChain ------------------------------------+
 * | [H] [ LionsPhil______ ] [ ] [ Player 4_______ ] |
 * | [ ] [ Player 2_______ ] [C] [ Player 5_______ ] |
 * | [ ] [ Player 3_______ ] [C] [ Player 6_______ ] |
 * |                                                 |
 * |     Grid size: [ _11 ] x [ __7 ]                |
 * |                                                 |
 * |                [ PLAY ] [ QUIT ]                |
 * +-------------------------------------------------+ */
/* But, realistically? Doing GUIs with ncurses is horrible. :( */

void do_gpl_waffle(void) {
	printw(
	"FreeChain, copyright (C) 2007--2010 Philip Boulain.\n"
	"FreeChain comes with ABSOLUTELY NO WARRANTY. This is free software, and you are welcome to redistribute it under certain conditions.\n"
	"For details see LICENSE.TXT, containing the text of the GNU GPL v2.\n"
	"\n");
}

