/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#include <stdlib.h>

#include <glade/glade.h>

#include "gtkloader.h"

/* Theoretically, should let this be overridable by environment variable. */
#ifndef DATADIR
#define DATADIR "./"
#endif

GdkPixbuf* load_colour_atom(const gchar* shape, const GdkColor* colour,
	uint8_t count, int width, int height) {

	const gchar* countstr = "";
	gchar* fn;
	GdkPixbuf* atom;
	GError* err = 0;
	guchar* pixels;
	guchar* p;
	int stride;
	int channels;
	double red, green, blue;

	/* Load the atom */
	switch(count) {
		case 0: countstr = "explode"; break;
		case 1: countstr = "1"; break;
		case 2: countstr = "2"; break;
		case 3: countstr = "3"; break;
		case 4: countstr = "4"; break;
		default: g_critical("Unexpected number of atoms %u", count);
	}
	fn = g_strconcat(DATADIR, "atomgraphics/",
		shape, "/", countstr, ".svg", NULL);
	if(!fn) { g_critical("Out of memory."); return 0; }

	atom = gdk_pixbuf_new_from_file_at_scale(fn, width, height, TRUE, &err);
	if(err) {
		g_critical("Cannot load atom '%s': %s.", fn, err->message);
		g_error_free(err);
		g_free(fn);
		return 0;
	}
	g_return_val_if_fail(atom, 0);
	g_free(fn);

	/* Colourise the atom using a shonky lighting algorithm.
	 * (red = diffuse, green = specular, blue = ignored) */
	red   = colour->red   / 65535.0;
	green = colour->green / 65535.0;
	blue  = colour->blue  / 65535.0;
	g_return_val_if_fail(
		gdk_pixbuf_get_colorspace(atom) == GDK_COLORSPACE_RGB, atom);
	g_return_val_if_fail(gdk_pixbuf_get_bits_per_sample(atom) == 8, atom);
	g_return_val_if_fail(gdk_pixbuf_get_has_alpha(atom), atom);
	pixels = gdk_pixbuf_get_pixels(atom);
	/* Size may have varied from request due to aspect ratio preservation */
	width    = gdk_pixbuf_get_width(atom);
	height   = gdk_pixbuf_get_height(atom);
	stride   = gdk_pixbuf_get_rowstride(atom);
	channels = gdk_pixbuf_get_n_channels(atom);
	g_return_val_if_fail(channels == 4, atom);

	for(int y = 0; y < height; y++) {
		p = pixels + (y * stride);
		for(int x = 0; x < width; x++) {
			uint16_t val;
			guchar diffuse  = p[0];
			guchar specular = p[1];
			val = (red   * diffuse) + specular;
			p[0] = val > 255 ? 255 : val;
			val = (green * diffuse) + specular;
			p[1] = val > 255 ? 255 : val;
			val = (blue  * diffuse) + specular;
			p[2] = val > 255 ? 255 : val;
			/* p[3] is alpha; leave it alone */
			p += channels;
		}
	}

	return atom;
}

static GladeXML* loaded_ui = 0;
static GladeXML* load_glade_ui(void) {
	if(!loaded_ui) {
		loaded_ui = glade_xml_new(DATADIR "gtkui.glade", NULL, NULL);
	}
	if(!loaded_ui) {
		g_critical("Couldn't load '" DATADIR "gtkui.glade'.");
		exit(1);
	}
	return loaded_ui;
}

GtkWidget* load_game_window(void) {
	return glade_xml_get_widget(load_glade_ui(), "gamewindow");
}

GtkTable* load_game_grid(void) {
	return GTK_TABLE(glade_xml_get_widget(load_glade_ui(), "gamegrid"));
}

