/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#include <stdlib.h>
#include "game.h"

#define CELL(x,y) ((x)+((y)*(game->width)))
#define XYBAD   ((x >= game->width) || (y >= game->height))
#define PLAYBAD (player >= game->players)

/* Add an atom to a cell, taking over ownership. Used by explode(). */
static void inflate(fcgame* game, uint8_t x, uint8_t y, uint8_t player) {
	fccell* cell = &(game->grid[CELL(x, y)]);
	/* Subtract cell's value from previous owner (may be zero) */
	game->totals[cell->player] -= cell->atoms;
	/* Modify the cell */
	cell->player = player;
	cell->atoms++;
	/* Add value to new owner */
	game->totals[cell->player] += cell->atoms;
}

/* (Possibly) blow up a cell. Works with game_play(). */
static void explode(fcgame* game, uint8_t x, uint8_t y) {
	bool n, e, s, w;
	fcexplosion splode;
	/* Test for instability */
	if(game->grid[CELL(x, y)].atoms <=
	   game_cellmax(game, x, y)) {
		return; /* Safe: base case */
	}
	/* Find available directions to explode in */
	n = (y != 0);
	e = (x < game->width-1);
	s = (y < game->height-1);
	w = (x != 0);
	/* Blow it up! */
	splode.player    = game->grid[CELL(x, y)].player;
	splode.magnitude = game->grid[CELL(x, y)].atoms; /* Save this... */
	game->grid[CELL(x, y)].atoms = 0; /* ...all gone now... */
	game->totals[splode.player] -= splode.magnitude; /* ...bye bye! */
	if(n) { inflate(game, x  , y-1, splode.player); }
	if(e) { inflate(game, x+1, y  , splode.player); }
	if(s) { inflate(game, x  , y+1, splode.player); }
	if(w) { inflate(game, x-1, y  , splode.player); }
	/* Tell the UI */
	splode.x = x;
	splode.y = y;
	splode.n = n ? game->grid[CELL(x  , y-1)].atoms : 0;
	splode.e = e ? game->grid[CELL(x+1, y  )].atoms : 0;
	splode.s = s ? game->grid[CELL(x  , y+1)].atoms : 0;
	splode.w = w ? game->grid[CELL(x-1, y  )].atoms : 0;
	game->callbacks->explode(game->callbacks->data, &splode);
	/* Did that eliminate someone? */
	for(uint8_t p = 0; (p < game->players) && !game->over; p++) {
		if(game->alive[p] && game->totals[p] == 0) {
			game->alive[p] = false;
			game->callbacks->eliminated(game->callbacks->data, p);
			game->survivors--;
		}
	}
	/* Did that eliminate everyone but this player? */
	if((game->survivors == 1) && !game->over) {
		game->callbacks->wins(game->callbacks->data, splode.player);
		game->over = true;
	}
	/* Recurse onto the neighbours */
	if(n && !game->over) { explode(game, x  , y-1); }
	if(e && !game->over) { explode(game, x+1, y  ); }
	if(s && !game->over) { explode(game, x  , y+1); }
	if(w && !game->over) { explode(game, x-1, y  ); }
}

/* Called automatically on victory or abort. */
static void game_destroy(fcgame* game) {
	free(game->alive);
	free(game->totals);
	free(game->grid);
}

bool game_init(fcgame* game, uint8_t width, uint8_t height, uint8_t players,
	const fccallbacks* callbacks) {

	if(!width || !height || !players) { return false; } /* Silly */
	if((width * height) < players) { return false; } /* Jam 1st round */
	game->width = width;
	game->height = height;
	game->players = players;
	game->curplay = 0;
	game->survivors = players;
	game->callbacks = callbacks;
	game->over = false;
	game->broken = false;
	game->grid = malloc(sizeof(fccell) * width * height);
	if(!game->grid) { return false; }
	game->totals = malloc(sizeof(uint32_t) * players);
	if(!game->totals) { free(game->grid); return false; }
	game->alive = malloc(sizeof(bool) * players);
	if(!game->alive) { free(game->grid); free(game->totals); return false; }

	for(uint8_t y = 0; y < game->height; y++) {
		for(uint8_t x = 0; x < game->width; x++) {
			game->grid[CELL(x, y)].atoms = 0;
			/* Shouldn't matter, as atoms == 0, but define. */
			game->grid[CELL(x, y)].player = 0;
		}
	}
	for(uint8_t p = 0; p < players; p++) {
		game->totals[p] = 0;
		game->alive[p] = true;
	}
	return true;
}

bool game_copy(const fcgame* src, fcgame* dst, const fccallbacks* callbacks) {
	if(src->over) { return false; }
	if(!game_init(dst, src->width, src->height, src->players,
		callbacks ? callbacks : src->callbacks)) { return false; }
	dst->curplay = src->curplay;
	dst->survivors = src->survivors;
	for(uint16_t cell = 0; cell < (src->height * src->width); cell++) {
		dst->grid[cell].atoms =
			src->grid[cell].atoms;
		dst->grid[cell].player =
			src->grid[cell].player;
	}
	for(uint8_t p = 0; p < src->players; p++) {
		dst->totals[p] = src->totals[p];
		dst->alive[p]  = src->alive[p];
	}
	return true;
}

/* Test to abort; use after every callback. */
#define MAYBEABORT() if(game->over) { goto finally; }

void game_play(fcgame* game) {
	/* (Historically, curplay was scoped here, but that made this non
	 * re-entrant, which was incompatable with game_break(). */
	/* Enter an infinite loop until broken out. */
	MAYBEABORT(); /* Simulator may abort before play. */
	while(1) {
		uint8_t x, y;
		bool good;

		/* Get an atom placement */
		good = false;
		while(!good) {
			game->callbacks->place(game->callbacks->data,
				game->curplay, &x, &y);
			MAYBEABORT();
			if(game_isvalid(game, game->curplay, x, y)) {
				good = true;
				game->callbacks->placed(game->callbacks->data,
					game->curplay, x, y);
				MAYBEABORT();
				if(game_isfull(game, x, y)) {
					game->callbacks->full(
						game->callbacks->data,
						game->curplay, x, y);
					MAYBEABORT();
				}
			} else {
				game->callbacks->invalid(game->callbacks->data,
					game->curplay, x, y);
				MAYBEABORT();
			}
		}

		/* Add atom here */
		inflate(game, x, y, game->curplay);

		/* Run explosions (recursive; early termination on abort) */
		explode(game, x, y);
		MAYBEABORT();

		/* Next player, looping, and skipping dead ones.
		 * (No players can be eliminated during the first round.) */
		do {
			game->curplay++;
			if(game->curplay == game->players)
				{ game->curplay = 0; }
		} while(!game->alive[game->curplay]);
	}
	
finally: /* Mmm...goto. Because you can't have a function which pops. */
	if(game->broken) {
		/* Actually, everything is fine; just popping the loop. */
		game->over = false;
		game->broken = false;
	} else {
		/* We're outta here. */
		game_destroy(game);
	}
	return;
}

#undef MAYBEABORT

void game_abort(fcgame* game) {
	game->over = true;
}

void game_break(fcgame* game) {
	game_abort(game);
	game->broken = true;
}

uint8_t game_width(const fcgame* game) { return game->width; }
uint8_t game_height(const fcgame* game) { return game->height; }
uint8_t game_players(const fcgame* game) { return game->players; }
uint8_t game_curplayer(const fcgame* game) { return game->curplay; }

const fccell* game_getcell(const fcgame* game, uint8_t x, uint8_t y) {
	if(XYBAD) return 0;
	return &(game->grid[CELL(x, y)]);
}

uint32_t game_gettotal(const fcgame* game, uint8_t player) {
	if(PLAYBAD) return 0;
	return game->totals[player];
}

bool game_alive(const fcgame* game, uint8_t player) {
	if(PLAYBAD) return false;
	return game->alive[player];
}

uint8_t game_cellmax(const fcgame* game, uint8_t x, uint8_t y) {
	uint8_t max = 3;
	if(XYBAD) return 0;
	if((y == 0) || (y == (game->height-1))) { max--; }
	if((x == 0) || (x == (game->width -1))) { max--; }
	return max;
}

bool game_isalmostfull(const fcgame* game, uint8_t x, uint8_t y) {
	if(XYBAD) return false;
	return (game_cellmax(game, x, y) ==
		(game->grid[CELL(x, y)].atoms + 1));
}

bool game_isfull(const fcgame* game, uint8_t x, uint8_t y) {
	if(XYBAD) return false;
	/* Cells in the process of exploding are counted as full */
	return (game_cellmax(game, x, y) <=
		game->grid[CELL(x, y)].atoms);
}

bool game_isvalid(const fcgame* game, uint8_t player, uint8_t x, uint8_t y) {
	if(!XYBAD && /* In range and... */
		/* ...either cell is unoccupied... */
		((game->grid[CELL(x, y)].atoms == 0) ||
		/* ...or occupied by that player. */
		 (game->grid[CELL(x, y)].player == player))) {
	
		return true;
	} else { return false; }
}

