/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#include <stdlib.h>
#include "ai.h"
#include "simulate.h"

/* A rule about FreeChain as a game space:
 *  - Placement on a full cell is isomorphic to placement on any adjacent full
 *    cell. (This applies recursively: full cells are isomorphic blocks.)
 * And an observation used to reduce the search space if possible:
 *  - Placement on a non-full cell adjacent to an enemy full cell is unwise.
 */

typedef struct {
	uifeedback* uif;
	fcgame* game;
	uint8_t player;
	uint8_t depth;
} minimaxstate;

/* Returns true if this move (assumed valid) is dangerous, defined as:
 *  - it is on a non-full cell AND
 *  - it is adjacent to at least one full enemy cell */
static bool dangerous(const fcgame* game,
	uint8_t player, uint8_t x, uint8_t y) {

	uint8_t w, h;

	if(game_isfull(game, x, y)) { return false; }

	w = game_width( game);
	h = game_height(game);

	/* Enemy cells are easily identified, as they are invalid moves. */
	return
		((x >   0) && !game_isvalid(game, player, x-1, y)
		                     && game_isfull(game, x-1, y)) ||
		((x < w-1) && !game_isvalid(game, player, x+1, y)
		                     && game_isfull(game, x+1, y)) ||
		((y >   0) && !game_isvalid(game, player, x, y-1)
		                     && game_isfull(game, x, y-1)) ||
		((y < h-1) && !game_isvalid(game, player, x, y+1)
		                     && game_isfull(game, x, y+1));
}

/* Last-gasp any-valid-move finder for if memory exhausted. */
static void mm_lastgasp(minimaxstate* st, uint8_t player, uint8_t w, uint8_t h,
	uint8_t* x, uint8_t* y) {

	st->uif->say(st->uif->data, "Yikes, memory exhausted!");
	for(uint8_t cy = 0; cy < h; cy++) {
		for(uint8_t cx = 0; cx < w; cx++) {
			if(game_isvalid(st->game, player, cx, cy))
				{ *x = cx; *y = cy; return; }
		}
	}
	/* Can never get here unless the game has also gone wrong. */
}

/* Co-routines: */
/* Evaluate a particular game state to a given search depth for a given player.
 * If depth is zero, uses a simple heuristic of atom count, less enemy atoms.
 * (Maximum value is beyond maximum number of atoms in a maximally-sized game:
 *  (3 * 254 * 254) + (2 * 254 * 4) + 4 = 195584) */
static int32_t mm_evaluate(aistate* state, simulator* sim,
	uint8_t player, uint8_t depth);
static const int32_t VAL_VICTORY =  200000;
static const int32_t VAL_DEFEAT  = -200000;
/* Find the best move to make for the given player, in the given sim. state,
 * using the given search depth (passed through to evaluate()). Chooses
 * randomly between equally good alternatives. Also returns value of move. */
static void mm_findbest(aistate* state, simulator* sim,
	uint8_t player, uint8_t depth, uint8_t* x, uint8_t* y, int32_t* val) {

	bool* validcells;
	bool optiunfull;
	uint8_t w, h;
	const fcgame* sgame; /* Simulated game; c.f. st->game, which is NOW */
	minimaxstate* st = (minimaxstate*) state;

	sgame = sim_game(sim);
	*val = 0; /* In case we have to lastgasp */
	w = game_width( st->game); /* Assumes that grid doesn't resize ;) */
	h = game_height(st->game);
	validcells = malloc(sizeof(bool) * w * h);
	if(!validcells) { mm_lastgasp(st, player, w, h, x, y); return; }
#define CELLVALID(X,Y) validcells[((Y)*w) + (X)]
#define CAN_L (cx > 0)
#define CAN_U (cy > 0)
#define CAN_R (cx < w-1)
#define CAN_D (cy < h-1)

	/* Compute all valid placements for the player */
	for(uint8_t cy = 0; cy < h; cy++) {
		for(uint8_t cx = 0; cx < w; cx++) {
			CELLVALID(cx, cy) = game_isvalid(sgame,
				player, cx, cy);
		}
	}

	/* Remove isomorphic moves (optimisation of search space) */
	for(uint8_t cy = 0; cy < h; cy++) {
		for(uint8_t cx = 0; cx < w; cx++) {
			if(CELLVALID(cx, cy) && game_isfull(sgame, cx, cy)) {
				/* Working right and down, therefore should
				 * never be true if a cell to the up or left
				 * is also true. */
				if(CAN_U && CELLVALID(cx, cy-1) &&
					game_isfull(sgame, cx, cy-1))
					{ CELLVALID(cx, cy) = false; }
				if(CAN_L && CELLVALID(cx-1, cy) &&
					game_isfull(sgame, cx-1, cy))
					{ CELLVALID(cx, cy) = false; }
/* XXX				if(CAN_R && CELLVALID(cx+1, cy) &&
					game_isfull(sgame, cx+1, cy))
					{ CELLVALID(cx+1, cy) = false; }
				if(CAN_D && CELLVALID(cx, cy+1) &&
					game_isfull(sgame, cx, cy+1))
					{ CELLVALID(cx, cy+1) = false; } */
			}
		}
	}

	/* Find at least one cell which is not dangerous. This lets us know that
	 * we can optimise by removing all dangerous cells (never wise). */
	optiunfull = false;
	for(uint8_t cy = 0; !optiunfull && cy < h; cy++) {
		for(uint8_t cx = 0; !optiunfull && cx < w; cx++) {
			if(CELLVALID(cx, cy) && !dangerous(sgame,
				player, cx, cy)) {

				optiunfull = true;
			}
		}
	}
	if(optiunfull) {
		for(uint8_t cy = 0; cy < h; cy++) {
			for(uint8_t cx = 0; cx < w; cx++) {
				if(CELLVALID(cx, cy) && dangerous(sgame,
					player, cx, cy)) {

					CELLVALID(cx, cy) = false;
				}
			}
		}
	}

	/* XXX Work out how to apply alpha-beta pruning to n-way minimax, and
	 * do it. It might help to sort candidate moves by trying full cells
	 * first. */
	/* Pick a bestval we are guaranteed to improve upon, so that we will
	 * make a valid move, even if our only valid move is to lose. */
	*x = 0; *y = 0;
	*val = VAL_DEFEAT - 1;
	/* FOR NOW do a dumb, exhaustive scan. */
	for(uint8_t cy = 0; cy < h; cy++) {
		for(uint8_t cx = 0; cx < w; cx++) {
			if(CELLVALID(cx, cy)) {
				int32_t tryval;
				simulator whatif;
				if(!sim_deepen(sim, &whatif, cx, cy)) {
					/* Invalid (should never happen, haha)
					 * or out of memory. */
					mm_lastgasp(st, player, w, h, x, y);
					return;
				}

				/* How good is that move for US? */
				tryval = mm_evaluate(state, &whatif, player,
					depth); /* depth decs. in eval. */
				if(tryval > *val) {
					*x = cx; *y = cy; *val = tryval;
				} else if(tryval == *val) {
					/* Randomly do this equally-good thing
					 * instead. This makes the combined AI
					 * behaviour (once we get co-recursing
					 * with evaluate()) non-deterministic,
					 * but that's interesting, right? */
					if(rand() < (RAND_MAX / (w*h))) {
						*x = cx; *y = cy;
						/* XXX I think this distribution
						 * will be weirdly biased. */
					}
				}

				sim_destroy(&whatif);
			}
		}
	}

#undef CELLVALID
#undef CAN_L
#undef CAN_R
#undef CAN_U
#undef CAN_D
	free(validcells);
	return;
}

static int32_t mm_evaluate(aistate* state, simulator* sim,
	uint8_t player, uint8_t depth) {

	if(sim_won(sim)) { /* Use constant */
		return (player == sim_player(sim)) ? VAL_VICTORY : VAL_DEFEAT;
	} else if(depth == 0) { /* Use heuristic */
		uint8_t players;
		int32_t value = 0;

		players = game_players(sim_game(sim));
		for(uint8_t p = 0; p < players; p++) {
			uint32_t atoms = game_gettotal(sim_game(sim), p);
			if(p == player) {
				value += atoms; /* Our atoms are good */
			} else {
				value -= atoms; /* Other atoms are bad */
			}
		}
		return value;
	} else { /* Assume that everyone will make their best possible moves.
	            Evaluate the final result for us. This branches horribly,
	            but that's minimax without pruning for you. */
		uint8_t x, y;
		int32_t val;

		/* Find the best move for the player whose go it will be. */
		mm_findbest(state, sim, sim_player(sim), depth-1, &x, &y, &val);

		/* If that player is us, that value is the value of this state.
		 * If it's not, it's how well our opponent can do, so negate. */
		return (sim_player(sim) == player) ? val : -val;
	}
}

static void mm_destruct(aistate* state) {
	free(state);
}

/*** AI callbacks ***/
static void mm_place(aistate* state, uint8_t* x, uint8_t* y) {
	int32_t val;
	simulator sim;
	minimaxstate* st = (minimaxstate*) state;

	if(sim_init(&sim, st->game)) {
		mm_findbest(state, &sim, st->player, st->depth, x, y, &val);
	} else { /* Crikey. Er, anything, then. */
		mm_lastgasp(st, st->player,
			game_width(st->game), game_height(st->game),
			x, y);
	}
}

static void mm_lost(aistate* state) {
	mm_destruct(state);
}

static void mm_won(aistate* state) {
	mm_destruct(state);
}

static void mm_broken(aistate* state) {
	mm_destruct(state);
}

static void mm_abort(aistate* state) {
	mm_destruct(state);
}

bool ai_minimax(aistate* state, aijumps* jumps, uifeedback* uif, fcgame* game,
	uint8_t player, uint8_t depth) {

	minimaxstate* mms = malloc(sizeof(minimaxstate));
	if(!mms) { return false; }
	*state = mms; /* Assigning to void**, remember */
	mms->uif    = uif;
	mms->game   = game;
	mms->player = player;
	mms->depth  = depth;
	jumps->place  = mm_place;
	jumps->lost   = mm_lost;
	jumps->won    = mm_won;
	jumps->broken = mm_broken;
	jumps->abort  = mm_abort;
	return true;
}

