/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

/* ncurses interface; huge thanks to the people behind this fine documentation:
 * http://tldp.org/HOWTO/NCURSES-Programming-HOWTO/ */

/* Yes, I'd like to ACTUALLY HAVE nanosleep, thanks. */
#define _POSIX_C_SOURCE 199309

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <ncurses.h>
#include "game.h"
#include "ai.h"
#include "ncursesui.h"

/* COLPLAY is defined in the header such that the setup UI can use it */
#define COLDEAD 7
#define COLGRID 8
#define COLSPLODE 9
#define COLPOPUP 10
#define COLPOPPRESS 11

/* Zero to four atoms; explosion. */
static const char atom_text[][4] = {"   "," o ","o o","ooo","o8o",">X<"};
static const uint8_t cell_width  = 4; /* Atoms plus borders on ONE side */
static const uint8_t cell_height = 2;

typedef struct {
	/* The game being played (not const; may abort) */
	fcgame* game;
	/* All information on how the game was configured; also needed to give
	 * AIs their state back. This is sort-of-const, but AIs may modify their
	 * state, and AI players may be unceremoniously renamed if broken. */
	setupinfo* gamesetup;
	/* Cursor position */
	uint8_t cx, cy;
	/* Player currently playing */
	uint8_t playing;
	/* Display caches */
	uint8_t longest_name; /* Not including NULL */
	int origin_x, origin_y; /* Top left of cell 0,0 */
	int player_x; /* Left edge of player list; 0 if disabled */
} playinfo;

/* Saner wrapping for the horrors of nanosleep */
static void centisleep(long centiseconds) {
	struct timespec s, r;
	s.tv_sec  = centiseconds / 100;
	s.tv_nsec = (centiseconds % 100) * 10000000;
	while(nanosleep(&s, &r) == EINTR) {
		s.tv_sec  = r.tv_sec;
		s.tv_nsec = r.tv_nsec;
	}
}


/** Rendering the game (only explosions and popups refresh() on their own) ****/

static void draw_recache(playinfo* play) {
	int total_width;
	int total_height;
	int size_x, size_y;
	getmaxyx(stdscr, size_y, size_x);
	play->player_x = 1;
	total_width = (game_width(play->game) * cell_width) + 2 + /* bdr+' ' */
		play->longest_name + 4; /* three digits + space */
	if(total_width > size_x) {
		/* Not enough room for player list; disable */
		play->player_x = 0;
		total_width = (game_width(play->game) * cell_width);
	}
	total_height = game_height(play->game) * cell_height;
	play->origin_x = (size_x - total_width ) / 2;
	play->origin_y = (size_y - total_height) / 2;
	/* If origin offscreen, screen too small; should warn here. :/ */
	if(play->origin_x < 0) { play->origin_x = 0; }
	if(play->origin_y < 0) { play->origin_y = 0; }
	if(play->player_x) {
		play->player_x = play->origin_x +
			(game_width(play->game) * cell_width) + 2;
	}
}

/* If cursor is false, it is suppressed; else, it is drawn in place. */
static void draw_cell(const playinfo* play,
	uint8_t x, uint8_t y, bool cursor) {

	const fccell* cell;
	bool highlight, n, e, s, w;
	int border_attr;
	int atoms;
	int dx, dy; /* Drawing co-ordinates */
	dx = play->origin_x + (x * cell_width);
	dy = play->origin_y + (y * cell_height);
	highlight = (cursor && x == play->cx && y == play->cy);

	if(highlight) {
		border_attr = COLOR_PAIR(
			COLPLAY(play->gamesetup->player_color[play->playing]));
		attron(A_BOLD);
	} else {
		border_attr = COLOR_PAIR(COLGRID);
	}
	attron(border_attr);

#define BORDERCHAR(x) (highlight ? ACS_CKBOARD : (x))
	/* Horizontal lines */
	for(uint8_t i = 1; i <= cell_width-1; i++) {
		mvaddch(dy,             dx+i, BORDERCHAR(ACS_HLINE));
		mvaddch(dy+cell_height, dx+i, BORDERCHAR(ACS_HLINE));
	}
	/* Vertical lines */
	mvaddch(dy+1, dx,            BORDERCHAR(ACS_VLINE));
	mvaddch(dy+1, dx+cell_width, BORDERCHAR(ACS_VLINE));
	/* Corners */
	n = (y != 0);
	e = (x < game_width( play->game)-1);
	s = (y < game_height(play->game)-1);
	w = (x != 0);
	mvaddch(dy, dx,
		n ? (w ? BORDERCHAR(ACS_PLUS) : BORDERCHAR(ACS_LTEE)    ) :
	            (w ? BORDERCHAR(ACS_TTEE) : BORDERCHAR(ACS_ULCORNER)));
	mvaddch(dy, dx+cell_width,
		n ? (e ? BORDERCHAR(ACS_PLUS) : BORDERCHAR(ACS_RTEE)    ) :
	            (e ? BORDERCHAR(ACS_TTEE) : BORDERCHAR(ACS_URCORNER)));
	mvaddch(dy+cell_height, dx+cell_width,
		s ? (e ? BORDERCHAR(ACS_PLUS) : BORDERCHAR(ACS_RTEE)    ) :
	            (e ? BORDERCHAR(ACS_BTEE) : BORDERCHAR(ACS_LRCORNER)));
	mvaddch(dy+cell_height, dx,
		s ? (w ? BORDERCHAR(ACS_PLUS) : BORDERCHAR(ACS_LTEE)    ) :
	            (w ? BORDERCHAR(ACS_BTEE) : BORDERCHAR(ACS_LLCORNER)));
#undef BORDERCHAR

	attroff(border_attr); if(highlight) { attroff(A_BOLD); }

	/* Cell contents */
	cell = game_getcell(play->game, x, y);
	atoms = cell->atoms;
	if(atoms > 4) { atoms = 4; }
	attron( COLOR_PAIR(
		COLPLAY(play->gamesetup->player_color[cell->player])));
	if(game_isfull(play->game, x, y)) { attron( A_BOLD); }
	mvprintw(dy+1, dx+1, atom_text[atoms]);
	if(game_isfull(play->game, x, y)) { attroff(A_BOLD); }
	attroff(COLOR_PAIR(
		COLPLAY(play->gamesetup->player_color[cell->player])));
}

static void draw_players(const playinfo* play) {
	int score_x = play->player_x + play->longest_name+1;
	if(!play->player_x) { return; }
	for(int p = 0; p < play->gamesetup->players; p++) {
		int colour;
		bool emphasise = (play->playing == p);	
		if(emphasise) { attron( A_BOLD); }
		colour = COLOR_PAIR(game_alive(play->game, p) ?
			COLPLAY(play->gamesetup->player_color[p]) : COLDEAD);
		attron( colour);
		mvprintw(play->origin_y + p, play->player_x,
			play->gamesetup->player_name[p]);
		mvprintw(play->origin_y + p, score_x,
			"%3d", game_gettotal(play->game, p));
		attroff(colour);
		if(emphasise) { attroff(A_BOLD); }
	}
}

static void draw_explosion(const playinfo* play, const fcexplosion* explosion) {
	int dx, dy; /* Drawing co-ordinates */
	dx = play->origin_x + (explosion->x * cell_width);
	dy = play->origin_y + (explosion->y * cell_height);
	/* Draw explosion */
	attron( COLOR_PAIR(COLSPLODE));
	attron( A_BOLD);
	mvprintw(dy+1, dx+1, atom_text[5]);
	attroff(A_BOLD);
	attroff(COLOR_PAIR(COLSPLODE));
	refresh();
	centisleep(10);
	/* Draw effect on neighbouring cells */
	if(explosion->y != 0)
		{ draw_cell(play, explosion->x,   explosion->y-1, false); }
	if(explosion->x < game_width( play->game)-1)
		{ draw_cell(play, explosion->x+1, explosion->y,   false); }
	if(explosion->y < game_height(play->game)-1)
		{ draw_cell(play, explosion->x,   explosion->y+1, false); }
	if(explosion->x != 0)
		{ draw_cell(play, explosion->x-1, explosion->y,   false); }
	draw_players(play); /* ...and on the totals */
	refresh();
	centisleep(10);
	/* Redraw this cell, now empty */
	draw_cell(play, explosion->x, explosion->y, false);
	refresh();
}

/* Completely redraw everything, e.g. at start, or after popup or resize. */
static void draw_gamescreen(const playinfo* play) {
	clear();
	for(uint8_t y = 0; y < game_height(play->game); y++) {
		for(uint8_t x = 0; x < game_width(play->game); x++) {
			draw_cell(play, x, y, true);
		}
	}
	/* Redraw the cell with the cursor to fix Z-order */
	draw_cell(play, play->cx, play->cy, true);
	draw_players(play);
}

static const char* waitstr = "(press key)";
static const int   waitstrlen = 11;
/* 'Player' can be 255 to suppress. In this case play can be null.
 * If waitkey is false, waits for two seconds instead. */
static void draw_popup(playinfo* play, uint8_t player, const char* message,
	bool waitkey) {

	WINDOW* popup;
	int width, height, x, y, namewidth;
	if(!play) { player = 255; }
	namewidth = (player == 255) ? 0 :
		strlen(play->gamesetup->player_name[player]);
	width = strlen(message) + 4;
	if(namewidth > width) { width = namewidth; }
	if(waitkey && waitstrlen > width) { width = waitstrlen; }
	height = 3;
	if(namewidth) { height++; }
	if(waitkey)   { height++; }
	getmaxyx(stdscr, y, x);
	x = (x - width ) / 2; if(x < 0) { x = 0; }
	y = (y - height) / 2; if(y < 0) { y = 0; }

	popup = newwin(height, width, y, x);
	wattron(popup, COLOR_PAIR(COLPOPUP));
	wattron(popup, A_BOLD);
	/* NCurses is practicing some kind of psychological warfare on me
	 * whereby it treats spaces as a special case and makes setting the
	 * freaking background such that it /actually works/ a nightmare. */
	for(y = 1; y < height-1; y++) {
		/* Draw a pretty pattern and hide it with colours.
		 * Otherwise, adjacent spaces turn eachother black! Yay! */
		for(x = 1; x < width-1; x+=2) { mvwaddch(popup, y, x, ' '); }
		wattroff(popup, A_BOLD);
		for(x = 2; x < width-1; x+=2) { mvwaddch(popup, y, x, '.'); }
		wattron( popup, A_BOLD);
	}
	box(popup, 0, 0);
	y = 1;
	if(namewidth) {
		int nx = (width - namewidth) / 2;
		mvwprintw(popup, y, nx, play->gamesetup->player_name[player]);
		y++;
	}
	mvwprintw(popup, y, 2, message); y++;
	if(waitkey) {
		wattroff(popup, COLOR_PAIR(COLPOPUP));
		wattron( popup, COLOR_PAIR(COLPOPPRESS));
		mvwprintw(popup, y, 2, waitstr);
		wattroff(popup, COLOR_PAIR(COLPOPPRESS));
		wattron( popup, COLOR_PAIR(COLPOPUP));
	}
	wrefresh(popup);

	if(waitkey) {
		int ch;
		bool resize = false;
		/* Always read getch(), to wait for a key; but if that key
		 * is KEY_RESIZE, loop and wait for a /real/ key. */
		do { ch = getch(); if(ch == KEY_RESIZE) { resize = true;} }
		while(ch == KEY_RESIZE);
		/* Did some swine resize us while the dialog was up? */
		if(resize) {
			refresh();
			draw_recache(play);
		}
	} else { centisleep(200); }

	delwin(popup);

	if(play) { draw_gamescreen(play); }
		else { clear(); }
}

/* Attempts to update current cursor position based on the provided mouse
 * co-ordinates; does not actually draw anything, but uses similar code.
 * Returns true if the co-ordinates fall within the grid. */
static bool draw_demouse(playinfo* play, int x, int y) {
	/* Remove the grid offset */
	x -= play->origin_x;
	y -= play->origin_y;
	/* Reduce by cell sizes */
	x /= cell_width;
	y /= cell_height;
	/* Bounds check */
	if(x < 0 || x >= game_width( play->game)) { return false; }
	if(y < 0 || y >= game_height(play->game)) { return false; }
	/* Modify cursor */
	play->cx = x; play->cy = y; return true;
}

/** Playing the game (inc. callbacks) *****************************************/

static void cb_place(void* data, uint8_t player, uint8_t* x, uint8_t* y) {
	uint8_t ai_x, ai_y;
	int ch = 0;
	bool getout = false;
	playinfo* play = (playinfo*) data;
	play->playing = player;
	draw_players(play);

	/* If computer, decide where to place */
	if(play->gamesetup->player_ai[player]) {
		/* Draw cursor before we think, in case it takes a while */
		draw_cell(play, play->cx, play->cy, true);
		refresh();
		play->gamesetup->ai_jumps[player].place(
			play->gamesetup->ai_state[player],
			&ai_x, &ai_y);
	}

	while(!getout) {
		MEVENT squeak;
		uint8_t ox, oy;
		ox = play->cx; oy = play->cy;
		/* Show cursor position */
		draw_cell(play, ox, oy, true);
		refresh();
		/* Get input, which is faked if it's an AI playing */
		if(play->gamesetup->player_ai[player]) {
			/* The AI uses Nethack keys, as it's oldschool */
			centisleep(10);
			if(ox == ai_x) {
				if(oy == ai_y) { ch = ' '; }
				else if(oy < ai_y) { ch = 'j'; }
				else { ch = 'k'; }
			} else if(ox < ai_x) { ch = 'l'; }
			else { ch = 'h'; }
		} else { /* Real humans get to use real keypresses */
			ch = getch();
		}
		/* Handle input */
		switch(ch) {
			case ' ': case '\n': case KEY_ENTER:
				getout = true;
				break;
			case 'q': case 'Q': case KEY_CLOSE: case KEY_EXIT:
				getout = true;
				game_abort(play->game);
				/* Destruct live AIs */
				for(uint8_t dplayer = 0; dplayer <
					play->gamesetup->players; dplayer++) {

					if(play->gamesetup->player_ai[dplayer]
						&& game_alive(play->game,
							dplayer)) {

						play->gamesetup->
							ai_jumps[dplayer].
							abort(play->gamesetup->
							ai_state[dplayer]);
					}
				}
				break;
			case 'h': case 'H': case KEY_LEFT:
				if(play->cx) { play->cx--; } break;
			case 'j': case 'J': case KEY_DOWN:
				if(play->cy < game_height(play->game)-1)
					{ play->cy++; } break;
			case 'k': case 'K': case KEY_UP:
				if(play->cy) { play->cy--; } break;
			case 'l': case 'L': case KEY_RIGHT:
				if(play->cx < game_width(play->game)-1)
					{ play->cx++; } break;
			case KEY_MOUSE:
				/* Move the cursor to where the user clicked;
				 * if this is where the cursor already was,
				 * take that as confirmation. */
				if(getmouse(&squeak) == OK) {
					if(draw_demouse(play,
						squeak.x, squeak.y)) {
						if((ox == play->cx) &&
						   (oy == play->cy))
							{ getout = true; }
					}
				}
				break;
			case KEY_RESIZE: /* from ncurses' SIGWINCH handler */
				refresh();
				draw_recache(play);
				draw_gamescreen(play);
				refresh();
				break;
			default:
		/* Go around again and hope for something more useful */
				break;
		}

		/* Undraw the cursor at its old location if moved */
		if((ox != play->cx) || (oy != play->cy)) {
			draw_cell(play, ox, oy, false);
		}
	}
	/* Provide the selected square */
	*x = play->cx;
	*y = play->cy;
}

static void cb_placed(void* data, uint8_t player, uint8_t x, uint8_t y) {
	draw_players((playinfo*) data);
	draw_cell((playinfo*) data, x, y, false);
	refresh();
}

static void cb_invalid(void* data, uint8_t player, uint8_t x, uint8_t y) {
	playinfo* play = (playinfo*) data;
	if(play->gamesetup->player_ai[play->playing]) {
		draw_popup(play, player, "has gone wrong and is being replaced with a human", true);
		strcpy(play->gamesetup->player_name[play->playing], "Broken");
		play->gamesetup->ai_jumps[player].broken
			(play->gamesetup->ai_state[player]);
		play->gamesetup->player_ai[player] = false;
		draw_gamescreen(play); /* Wipe trailing name */
	} else {
		draw_popup(play, player, "can only place atoms on empty or self-owned squares", true);
	}
}

static void cb_full(void* data, uint8_t player, uint8_t x, uint8_t y) {}

static void cb_explode(void* data, const fcexplosion* explosion) {
	draw_explosion((playinfo*) data, explosion);
}

static void cb_eliminated(void* data, uint8_t player) {
	playinfo* play = (playinfo*) data;

	draw_players(play);
	refresh();
	draw_popup(play, player, "is out of the game!", false);
	if(play->gamesetup->player_ai[player]) {
		play->gamesetup->ai_jumps[player].lost
			(play->gamesetup->ai_state[player]);
	}
}

static void cb_wins(void* data, uint8_t player) {
	playinfo* play = (playinfo*) data;

	refresh();
	draw_popup(play, player, "has won the game!", true);
	if(play->gamesetup->player_ai[player]) {
		play->gamesetup->ai_jumps[player].won
			(play->gamesetup->ai_state[player]);
	}
}

static void fb_say(void* data, const char* phrase) {
	/* AI code only runs during the AI's turn, so they must be the current
	 * player. */
	playinfo* play = (playinfo*) data;
	draw_popup(play, game_curplayer(play->game), phrase, true);
}

static void fb_hlcell(void* data, uint8_t x, uint8_t y, int16_t value) {
	/* TODO Draw the values (cropped -255 < v < 255) in the cell borders */
}

static void fb_hlapply(void* data) {
	/* TODO Wait two seconds, then repaint the grid */
}

/* Play the game in raw ncurses. */
static void play_game(setupinfo* gamesetup) {
	playinfo play;
	fcgame game;
	fccallbacks callbacks;
	uifeedback feedback;
	
	/* Initialise callback data */
	play.game = &game;
	play.gamesetup = gamesetup;
	play.cx = play.cy = 0;
	play.playing = 0;
	play.longest_name = 0;
	for(uint8_t p = 0; p < gamesetup->players; p++) {
		size_t len = strlen(gamesetup->player_name[p]);
		if(len > play.longest_name) { play.longest_name = len; }
	}

	callbacks.data = &play;
	callbacks.place      = cb_place;
	callbacks.placed     = cb_placed;
	callbacks.invalid    = cb_invalid;
	callbacks.full       = cb_full;
	callbacks.explode    = cb_explode;
	callbacks.eliminated = cb_eliminated;
	callbacks.wins       = cb_wins;

	feedback.data = &play;
	feedback.say     = fb_say;
	feedback.hlcell  = fb_hlcell;
	feedback.hlapply = fb_hlapply;

	/* Hide the cursor */
	curs_set(0);

	/* Run the game */
	if(game_init(&game, gamesetup->width, gamesetup->height,
		gamesetup->players, &callbacks)) {

		bool aigood = true;
		/* Create AI players */
		for(uint8_t player = 0; aigood &&
			(player < gamesetup->players); player++) {

			if(gamesetup->player_ai[player]) {
				/* All AI players are minimax at the moment. */
				if(!ai_minimax(
					&gamesetup->ai_state[player],
					&gamesetup->ai_jumps[player],
					&feedback, &game, player, 2)) {

					draw_popup(0, 255,
			"Can't create AI player (out of memory?)", true);
					/* Sigh. Destruct all previous. */
					for(int16_t dplayer = player;
						dplayer > 0; dplayer--) {

						gamesetup->ai_jumps[dplayer].
					abort(&gamesetup->ai_state[dplayer]);
					}
					aigood = false;
				}
			}
		}


		if(aigood) {
			/* Draw initial state */
			draw_recache(&play);
			draw_gamescreen(&play);

			game_play(&game);
		}
	} else {
		draw_popup(0, 255,
			"Can't start game (out of memory?)", true);
	}

	/* Restore cursor */
	curs_set(1);
}

/** Startup and shutdown ******************************************************/

/* Set up ncurses */
static bool gfx_init(void) {
	initscr();
	if(!has_colors()) {
		endwin();
		fprintf(stderr,
			"Sorry, FreeChain requires a colour terminal.\n");
		return false;
	}
	noecho(); /* Don't display keyboard input unless told */
	keypad(stdscr, true); /* Allow arrow keys etc. */
	/* Valgrind is showing that this mallocs up a block which ncurses then
	 * reads beyond. As far as I can tell, my usage is correct, and that's
	 * just a bug in ncurses. Can't tell if it's been reported. */
	mousemask(BUTTON1_CLICKED, 0); /* Accept simple mouse clicks */
	start_color(); /* Use colour */
	/* There are six usable colours, hence the hard six-player limit */
	init_pair(COLPLAY(0), COLOR_RED,     COLOR_BLACK);
	init_pair(COLPLAY(1), COLOR_YELLOW,  COLOR_BLACK);
	init_pair(COLPLAY(2), COLOR_GREEN,   COLOR_BLACK);
	init_pair(COLPLAY(3), COLOR_CYAN,    COLOR_BLACK);
	init_pair(COLPLAY(4), COLOR_BLUE,    COLOR_BLACK);
	init_pair(COLPLAY(5), COLOR_MAGENTA, COLOR_BLACK);
	init_pair(COLDEAD,    COLOR_WHITE,   COLOR_BLACK); /* Grey */
	init_pair(COLGRID,    COLOR_WHITE,   COLOR_BLACK); /* Grey */
	init_pair(COLSPLODE,  COLOR_WHITE,   COLOR_BLACK);
	init_pair(COLPOPUP,   COLOR_WHITE,   COLOR_WHITE); /* On grey */
	init_pair(COLPOPPRESS,COLOR_BLACK,   COLOR_WHITE); /* On grey */
	clear();
	refresh();
	return true;
}

/* Shut down ncurses */
static void gfx_deinit(void) {
	endwin();
}

int main(int argc, char** argv) {
	setupinfo gamesetup;

	/* We need to do this on behalf of the AI, as we are the entry point. */
	srand(time(0));

	if(!gfx_init()) { return 1; }
	while(setup_game(&gamesetup)) {
		play_game(&gamesetup);
	}
	gfx_deinit();
	return 0;
}

/* In-game display (in colour):
 * +---+---+ Harry     7
 * | o |o o| Player 5  3
 * +---+---+ Player 6  0
 * |ooo|o8o| (usernames in colour, on right, if space; greyed if dead)
 * #####---+ (current player in bold; full cells in bold)
 * #   #>X<| (explosions flash in white)
 * #####---+ (selected cell border is bright player and thick; rest are grey)
 */

