/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#ifndef GAME_H_
#define GAME_H_

#include <stdint.h>
#include <stdbool.h>

#define DEFAULT_GRID_WIDTH 11
#define DEFAULT_GRID_HEIGHT 7

/* Opaque game state structure. The game logic does not use any static data,
 * and it is therefore safe to have multiple, concurrent games using this. */
typedef struct fcgame_s fcgame;

typedef struct {
	uint8_t player;
	uint8_t atoms;
} fccell;

typedef struct {
	/* Grid co-ordinates of the explosion. This cell will now be empty. */
	uint8_t x, y;
	/* Player who owns the exploding atom. They will now own all adjacent
	 * grid squares. */
	uint8_t player;
	/* Magnitude of the explosion. This is the number of atoms in the cell
	 * that exploded, and will thus be between two and four. */
	uint8_t magnitude;
	/* Number of atoms now in adjacent (north, east, south, west) cells.
	 * These will all be owned by the player. North is negative Y; west is
	 * negative X. Cells outside of the grid contain zero atoms. */
	uint8_t n, e, s, w;
} fcexplosion;

/* All callbacks must be populated! */
typedef struct {
	/* Private data pointer for your own use, e.g. if you want to support
	 * multiple simultaneous games. */
	void* data;
	/* Request for atom placement. X must fit within width; Y must fit
	 * within height; and the cell must not be occupied by opponent atoms.
	 * To make the N/E/S/W explosion fields less confusing, the grid should
	 * have its origin in the bottom-left corner. If the placement is
	 * invalid, the invalid() callback is called, followed by this again. */
	void (*place)(void* data, uint8_t player, uint8_t* x, uint8_t* y);
	/* Indicates that the atom was placed successfully; you should probably
	 * redraw this cell now. */
	void (*placed)(void* data, uint8_t player, uint8_t x, uint8_t y);
	/* Indicates that the place the player tried to place the atom is
	 * already occupied. Will be followed by a repeated place() request. */
	void (*invalid)(void* data, uint8_t player, uint8_t x, uint8_t y);
	/* Immediately follows a place() if that placement has caused the cell
	 * to become as full as is safe (further atoms will explode). */
	void (*full)(void* data, uint8_t player, uint8_t x, uint8_t y);
	/* An atom has exploded in this cell. You very likely want to display
	 * the explosion in some manner, and update the grid graphics. */
	void (*explode)(void* data, const fcexplosion* explosion);
	/* A player was eliminated (all of their atoms were destroyed). */
	void (*eliminated)(void* data, uint8_t player);
	/* A player has won (only their atoms remain). */
	void (*wins)(void* data, uint8_t player);
} fccallbacks;

/* Initialise a fcgame structure, ready for a new game. 'Players' is simply
 * a count, and player numbers will range from zero to one less. Expects the
 * callback structure to be around for at least as long as the game. Returns
 * true if succeeded; false if there was not enough memory. */
bool game_init(fcgame* game, uint8_t width, uint8_t height, uint8_t players,
	const fccallbacks* callbacks);

/* Copy a game state from one structure to another. This is useful for AI
 * players which wish to perform "what-if" simulations. You should almost
 * certainly provide a different callback structure, but you /may/ preserve
 * the old one by passing a NULL. The source game must not be over.
 * Note that this will necessarily initialise the destination structure, so
 * you should ensure that it is played at some point, even if you then
 * immediately abort it. May return false if insufficient memory. */
bool game_copy(const fcgame* src, fcgame* dst, const fccallbacks* callbacks);

/* Play the game. This will start invoking callbacks, and return once the
 * game has been either won or aborted. The game structure will then be
 * deinitialised. */
void game_play(fcgame* game);

/* Abort a game. No further callbacks will be called, and game_play() will
 * return. */
void game_abort(fcgame* game);

/* Break from a game. This is like abort(), but does not perform destruction.
 * It is also not generally very safe, as resuming with play() will RESTART the
 * current move, even if you broke halfway through explosions. Generally, it
 * exists for the simulator, and that is where it should be left. */
void game_break(fcgame* game);

/** Accessors (especially useful for AIs) **/
uint8_t game_width(const fcgame* game);
uint8_t game_height(const fcgame* game);
uint8_t game_players(const fcgame* game);

/* Get the player who is currently playing. This is updated before place()
 * calls; in other calls, it will be the player who has played, and whose
 * actions are now taking effect. Again, mostly for the simulator. */
uint8_t game_curplayer(const fcgame* game);

/* Retrieve information about a given cell. Useful for general-purpose screen
 * redrawing. */
const fccell* game_getcell(const fcgame* game, uint8_t x, uint8_t y);

/* Return the total number of atoms owned by a particular player. */
uint32_t game_gettotal(const fcgame* game, uint8_t player);

/* Is the given player alive? Players may legitimately have a total of zero
 * atoms yet still be alive if they have not yet moved. */
bool game_alive(const fcgame* game, uint8_t player);

/* Returns the maximum number of safe atoms a cell can hold. */
uint8_t game_cellmax(const fcgame* game, uint8_t x, uint8_t y);

/* True if a cell is one atom away from being full. Useful to tell in advance
 * if placement here will cause full() to be triggered, e.g. for UI hints. */
bool game_isalmostfull(const fcgame* game, uint8_t x, uint8_t y);

/* True if a cell is full. This means that further atoms there will cause an
 * explosion. Again, useful for UI hints. */
bool game_isfull(const fcgame* game, uint8_t x, uint8_t y);

/* Return true if it would be valid for the given player to place an atom
 * in the given cell. This doesn't check that it's actually their turn. */
bool game_isvalid(const fcgame* game, uint8_t player, uint8_t x, uint8_t y);

/******************************************************************************/
/* The game structure is opaque. However, to allow you to stack-allocate games,
 * the structure is here, so that the compiler knows how big it is. */
struct fcgame_s {
	uint8_t width;
	uint8_t height;
	fccell* grid;
	uint8_t players; /* number of players */
	uint8_t curplay; /* current player */
	uint8_t survivors; /* number of players alive */
	uint32_t* totals; /* player's atom count */
	bool* alive; /* player is alive (zero atoms = starting/freshly dead) */
	const fccallbacks* callbacks;
	bool over; /* Used to indicate aborts as well as victories. */
	bool broken; /* Indicates that the game is 'over' because of _break() */
};
/* Resist the temptation to manipulate this structure yourself. */
/******************************************************************************/

#endif

