/* FreeChain - A cross-UI, cross-platform, Free version of Chain Reaction     *
 * Copyright 2007 Philip Boulain. Licenced under the GNU GPL.                 */

#ifndef AI_H_
#define AI_H_

#include <stdint.h>
#include <stdbool.h>
#include "game.h"
#include "ui.h"

/* Generic AI interface. Initialising an AI's jump table and state is
 * implementation-specific, as are the details of that state. Initialisation
 * will at least requre the game (to query the board) and player number.
 * AI state is destructed when one of the lost, won, broken or aborted
 * callbacks are invoked. The UI will have initialised srandom for you. */

typedef void* aistate;

typedef struct {
	/* Request placement of an atom. AIs must always generate valid
	 * placements. */
	void (*place)(aistate* state, uint8_t* x, uint8_t* y);
	/* Tell the AI that it has lost. This is potentially useful for learning
	 * AIs, but also triggers deconstruction of its state. */
	void (*lost)(aistate* state);
	/* Tell the AI that it has won; like lost(), but a happier event. */
	void (*won)(aistate* state);
	/* Tell the AI that it has broken: it has tried to make an invalid move
	 * move and is being replaced with a human to avoid an infinite loop. */
	void (*broken)(aistate* state);
	/* Abort the game. This is a destruction with no particular semantics;
	 * possibly because the game was quit by a human. */
	void (*abort)(aistate* state);
} aijumps;

/*** AI implementations *******************************************************/
/* Conventionally, these will take a state and jump structure to initialise,
 * a UI feedback callback structure, a game to play in and a player to play as,
 * some implementation-specific parameters, and return true on success or
 * false on failure. */

bool ai_minimax(aistate* state, aijumps* jumps, uifeedback* uif, fcgame* game,
	uint8_t player, uint8_t depth);

/* XXX Really need a factory pattern here. */

#endif

